// $Id: CClassWriter.cpp,v 1.6 2007/02/11 02:20:05 paul Exp $

/*
 * All contents of this source code are copyright 2007 Exp Digital Uk.
 * This file forms part of the example projects for the Infinity API.
 * You may not redistribute either the source code or the compiled binary.
 * The Infinity API is covered by a fair use licence which you should have recieved with this
 * file. If you didnt receieve the licence, please contact us via http://www.expdigital.co.uk
 */

#include "CClassWriter.hpp"
#include <Basics/CDialog.hpp>
using Exponent::GUI::Basics::CDialog;
using Exponent::ClassWrite::CClassWriter;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CClassWriter, CCountedObject);

//	===========================================================================
CClassWriter::CClassWriter(const CString &name, const EClassType type, const EParentType parent, TStringCountedPointerArray &namespaces, const CSystemString &outputFolder)
			: m_currentTabIndent(0)
			, m_name(name)
			, m_classType(type)
			, m_parentType(parent)
			, m_outputPath(outputFolder)
{
	EXPONENT_CLASS_CONSTRUCTION(CClassWriter);

	for (long i = 0; i < namespaces.getArraySize(); i++)
	{
		m_namespace.addElement(namespaces.elementAtIndex(i));
	}

	// This next call removes all NULL entries and creates a tight array of only valid elements
	m_namespace.reorder();

	this->writeHeaderFile();
	this->writeImplementationFile();
}

//	===========================================================================
CClassWriter::~CClassWriter()
{
	EXPONENT_CLASS_DESTRUCTION(CClassWriter);
}

//	===========================================================================
void CClassWriter::writeHeaderFile()
{
	// Construct the header file path
	CSystemString headerPath = m_outputPath;
	headerPath.appendPath(m_name);
	headerPath.appendNewExtension("hpp");

	// Open the file stream
	if (!m_stream.openStream(headerPath, CTextStream::e_output))
	{
		CDialog::notifyUser("Failed to open the file stream, cannot continue", "CClassWrite error", true);
		return;
	}

	// Now write all the component parts
	this->writeHeaderGuard();
	this->writeIncludes();
	this->writeNamespaceOpening();
	this->writeDoxygenMainComment();
	this->writeClassDeclaration();
	this->writePublicOpener();
	this->writeConstructor();
	this->writeDestructor();
	this->writeCountedObjectFunctions();
	this->writeControlFunctions();
	this->writeProtectedOpener();
	this->writeTemplateExtras();
	this->writeNamespaceClosing();

	// Finally we close the stream
	m_stream.closeStream();
}

//	===========================================================================
void CClassWriter::writeTabs()
{
	for (long i = 0; i < m_currentTabIndent; i++)
	{
		m_stream << "\t";
	}
}

//	===========================================================================
void CClassWriter::writeHeaderGuard()
{
	m_stream << "#ifndef __" << m_name << "__\n";
	m_stream << "#define __" << m_name << "__\n\n";
}

//	===========================================================================
void CClassWriter::writeIncludes()
{
	switch(m_parentType)
	{
		case e_nothing:
			// We dont need to include anything
		break;
		case e_iCountedObject:
			m_stream << "#include <Basics/ICountedObject.hpp>\n";
			m_stream << "using Exponent::Basics::ICountedObject;\n\n";
		break;
		case e_cCountedObject:
			m_stream << "#include <Basics/CCountedObject.hpp>\n";
			m_stream << "using Exponent::Basics::CCountedObject;\n\n";
		break;
		case e_control:
			m_stream << "#include <Controls/CControl.hpp>\n";
			m_stream << "using Exponent::GUI::Controls::CControl;\n\n";
		break;
		case e_controlPanel:
			m_stream << "#include <Controls/CControlPanel.hpp>\n";
			m_stream << "using Exponent::GUI::Controls::CControlPanel;\n\n";
		break;
	}
}

//	===========================================================================
void CClassWriter::writeNamespaceOpening()
{
	m_currentTabIndent = -1;
	for (long i = 0; i < m_namespace.getArraySize(); i++)
	{
		// Get the namespace
		const CString *myNamespace = m_namespace.constElementAtIndex(i);

		// Check its valid
		if (myNamespace == NULL)
		{
			continue;
		}

		// Write the tabs to move it over
		m_currentTabIndent++;
		this->writeTabs();
		m_stream << "namespace " << *myNamespace << "\n";
		this->writeTabs();
		m_stream << "{\n";
	}
	m_currentTabIndent++;
}

//	===========================================================================
void CClassWriter::writeNamespaceClosing()
{
	for (long i = 0; i < m_namespace.getArraySize(); i++)
	{
		const CString *myNamespace = m_namespace.constElementAtIndex(i);

		if (myNamespace == NULL)
		{
			continue;
		}
		m_currentTabIndent--;
		this->writeTabs();
		if (i == m_namespace.getArraySize() - 1)
		{
			m_stream << "}\n#endif\t\t// End of " << m_name << ".hpp";
		}
		else
		{
			m_stream << "}\n";
		}
	}
}

//	===========================================================================
void CClassWriter::writeDoxygenMainComment()
{
	// Write class doxygen comments header
	this->writeTabs();
	m_stream << "/**\n";
	this->writeTabs();

	// Write the type and file information
	switch(m_classType)
	{
		case e_class:			m_stream << " * @class ";			break;
		case e_interface:		m_stream << " * @interface ";		break;
		case e_struct:			m_stream << " * @struct ";			break;
		case e_template:		m_stream << " * @class ";			break;
	}

	// Write the file information
	m_stream << m_name << " " << m_name << ".hpp\n";
	this->writeTabs();
	m_stream << " * @brief TODO FILL THIS IN\n";
	this->writeTabs();
	m_stream << " *\n";

	// Write the date of writing
	CTime myTime;
	this->writeTabs();
	m_stream << " * @date " << myTime.getDay() << "/" << myTime.getMonth() << "/" << myTime.getYear() << "\n";

	// Write the author
	this->writeTabs();
	m_stream << " * @author Paul Chana\n";

	// Write the version information
	this->writeTabs();
	m_stream << " * @version 1.0.0 Initial version\n";

	// Space and then the copytright
	this->writeTabs();
	m_stream << " * \n";
	this->writeTabs();
	m_stream << " * @note All contents of this source code are copyright " << myTime.getYear() << " Exp Digital Uk.\n";
	this->writeTabs();
	m_stream << " * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n";
	this->writeTabs();
	m_stream << " * with the source code. If you didnt, please refer to http://www.expdigital.co.uk\n";
	this->writeTabs();
	m_stream << " * All content is the Intellectual property of Exp Digital Uk.\n";
	this->writeTabs();
	m_stream << " * Certain sections of this code may come from other sources. They are credited where applicable.\n";
	this->writeTabs();
	m_stream << " * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk";
	this->writeTabs();
	m_stream << " * \n";

	// Write the CVS comments
	this->writeTabs();
	m_stream << " *\n";
	this->writeTabs();
	m_stream << " * $Id: CClassWriter.cpp,v 1.6 2007/02/11 02:20:05 paul Exp $\n";
	this->writeTabs();
	m_stream << " */\n";
}

//	===========================================================================
void CClassWriter::writeClassDeclaration()
{
	this->writeTabs();

	// First we write the class declaration
	switch(m_classType)
	{
		case e_class:		m_stream << "class "						   << m_name;			break;
		case e_interface:	m_stream << "interface "					   << m_name;			break;
		case e_template:	m_stream << "template <class TypeName> class " << m_name;			break;
		case e_struct:		m_stream << "struct "						   << m_name;			break;
	}

	// Now write the inheritance declaration
	switch(m_parentType)
	{
		case e_nothing:			m_stream << "\n";								break;
		case e_iCountedObject:	m_stream << " : public ICountedObject\n";		break;
		case e_cCountedObject:	m_stream << " : public CCountedObject\n";		break;
		case e_control:			m_stream << " : public CControl\n";				break;
		case e_controlPanel:	m_stream << " : public CControlPanel\n";		break;
	}

	// Opening brace
	this->writeTabs();
	m_stream << "{\n";
	m_currentTabIndent++;

	// Now write the exponent class declaration stuff, as required
	switch(m_parentType)
	{
		case e_nothing:
		case e_iCountedObject:
			m_stream << "\n";
		break;
		case e_cCountedObject:
		case e_control:
		case e_controlPanel:
			// Write the exponent class decl
			this->writeTabs();
			m_stream << "/** @cond */\n";
			this->writeTabs();
			m_stream << "EXPONENT_CLASS_DECLARATION;\n";
			this->writeTabs();
			m_stream << "/** @endcond */\n\n";
		break;
	}
	m_currentTabIndent--;
}

//	===========================================================================
void CClassWriter::writePublicOpener()
{
	if (m_classType == e_struct)
	{
		m_stream << "\n";
		this->writeTabs();
		m_stream << "\t// TODO: FILL THIS IN!";
		return;
	}
	m_stream << "//	===========================================================================\n\n";
	this->writeTabs();
	m_stream << "public:\n\n";
	m_stream << "//	===========================================================================\n\n";
}

//	===========================================================================
void CClassWriter::writeConstructor()
{
	m_currentTabIndent++;
	switch(m_classType)
	{
		case e_class:
			switch(m_parentType)
			{
				case e_nothing:
				case e_iCountedObject:
				case e_cCountedObject:
					this->writeTabs();
					m_stream << "/**\n";
					this->writeTabs();
					m_stream << " * Construction\n";
					this->writeTabs();
					m_stream << " */\n";
					this->writeTabs();
					m_stream << m_name << "();\n\n";
				break;
				case e_control:
					this->writeTabs();
					m_stream << "/**\n";
					this->writeTabs();
					m_stream << " * Construction\n";
					this->writeTabs();
					m_stream << " * @param root The root control that this control is being added to\n";
					this->writeTabs();
					m_stream << " * @param uniqueId The unique Identifier of this control or CCONTROL_NO_ID_REQUIRED if no id is required\n";
					this->writeTabs();
					m_stream << " * @param area The area of this control relative to the root controls top left\n";
					this->writeTabs();
					m_stream << " * @param listener The action listener\n";
					this->writeTabs();
					m_stream << " */\n";
					this->writeTabs();
					m_stream << m_name << "(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener = NULL);\n\n";
				break;
				case e_controlPanel:
					this->writeTabs();
					m_stream << "/**\n";
					this->writeTabs();
					m_stream << " * Construction\n";
					this->writeTabs();
					m_stream << " * @param parent THe parent window holding this control root\n";
					this->writeTabs();
					m_stream << " * @param root The root control that this control is being added to\n";
					this->writeTabs();
					m_stream << " * @param uniqueId The unique Identifier of this control or CCONTROL_NO_ID_REQUIRED if no id is required\n";
					this->writeTabs();
					m_stream << " * @param area The area of this control relative to the root controls top left\n";
					this->writeTabs();
					m_stream << " * @param listener The action listener\n";
					this->writeTabs();
					m_stream << " */\n";
					this->writeTabs();
					m_stream << m_name << "(IWindow *parent, IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener = NULL);\n\n";
				break;
			}
		break;
		case e_template:
			switch(m_parentType)
			{
				case e_nothing:
				case e_iCountedObject:
					this->writeTabs();
					m_stream << "/**\n";
					this->writeTabs();
					m_stream << " * Construction\n";
					this->writeTabs();
					m_stream << " */\n";
					this->writeTabs();
					m_stream << m_name << "()\n";
					this->writeTabs();
					m_stream << "{\n";
					m_currentTabIndent++;
					this->writeTabs();
					m_currentTabIndent--;
					m_stream << "// TODO: FILL THIS IN!\n";
					this->writeTabs();
					m_stream << "}\n\n";
				break;
				case e_cCountedObject:
					this->writeTabs();
					m_stream << "/**\n";
					this->writeTabs();
					m_stream << " * Construction\n";
					this->writeTabs();
					m_stream << " */\n";
					this->writeTabs();
					m_stream << m_name << "()\n";
					this->writeTabs();
					m_stream << "{\n";
					m_currentTabIndent++;
					this->writeTabs();
					m_currentTabIndent--;
					m_stream << "EXPONENT_CLASS_CONSTRUCTION(" << m_name << "<TypeName>);\n\n";
					m_currentTabIndent++;
					this->writeTabs();
					m_currentTabIndent--;
					m_stream << "// TODO: FILL THIS IN!\n";
					this->writeTabs();
					m_stream << "}\n\n";
				break;
				case e_control:
					this->writeTabs();
					m_stream << "/**\n";
					this->writeTabs();
					m_stream << " * Construction\n";
					this->writeTabs();
					m_stream << " * @param root The root control that this control is being added to\n";
					this->writeTabs();
					m_stream << " * @param uniqueId The unique Identifier of this control or CCONTROL_NO_ID_REQUIRED if no id is required\n";
					this->writeTabs();
					m_stream << " * @param area The area of this control relative to the root controls top left\n";
					this->writeTabs();
					m_stream << " * @param listener The action listener\n";
					this->writeTabs();
					m_stream << " */\n";
					this->writeTabs();
					m_stream << m_name << "(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener = NULL) : CControl(root, uniqueId, area, listener)\n";
					this->writeTabs();
					m_stream << "{\n";
					m_currentTabIndent++;
					this->writeTabs();
					m_currentTabIndent--;
					m_stream << "EXPONENT_CLASS_CONSTRUCTION(" << m_name << "<TypeName>);\n\n";
					m_currentTabIndent++;
					this->writeTabs();
					m_currentTabIndent--;
					m_stream << "// TODO: FILL THIS IN!\n";
					this->writeTabs();
					m_stream << "}\n\n";
				break;
				case e_controlPanel:
					this->writeTabs();
					m_stream << "/**\n";
					this->writeTabs();
					m_stream << " * Construction\n";
					this->writeTabs();
					m_stream << " * @param parent THe parent window holding this control root\n";
					this->writeTabs();
					m_stream << " * @param root The root control that this control is being added to\n";
					this->writeTabs();
					m_stream << " * @param uniqueId The unique Identifier of this control or CCONTROL_NO_ID_REQUIRED if no id is required\n";
					this->writeTabs();
					m_stream << " * @param area The area of this control relative to the root controls top left\n";
					this->writeTabs();
					m_stream << " * @param listener The action listener\n";
					this->writeTabs();
					m_stream << " */\n";
					this->writeTabs();
					m_stream << m_name << "(IWindow *parent, IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener = NULL) : CControlPanel(root->getParentWindow(), root, uniqueId, area, listener)\n";
					this->writeTabs();
					m_stream << "{\n";
					m_currentTabIndent++;
					this->writeTabs();
					m_currentTabIndent--;
					m_stream << "EXPONENT_CLASS_CONSTRUCTION(" << m_name << "<TypeName>);\n\n";
					m_currentTabIndent++;
					this->writeTabs();
					m_currentTabIndent--;
					m_stream << "// TODO: FILL THIS IN!\n";
					this->writeTabs();
					m_stream << "}\n\n";
				break;
			}
		break;
		case e_interface:
		case e_struct:
			// We dont have a constructor because we are an interface or struct
		break;
	}

	m_currentTabIndent--;
}

//	===========================================================================
void CClassWriter::writeDestructor()
{
	m_currentTabIndent++;
	switch(m_classType)
	{
		case e_class:
			this->writeTabs();
			m_stream << "/**\n";
			this->writeTabs();
			m_stream << " * Destruction\n";
			this->writeTabs();
			m_stream << " */\n";
			this->writeTabs();
			m_stream << "virtual ~" << m_name << "();\n\n";
		break;
		case e_interface:
			this->writeTabs();
			m_stream << "/**\n";
			this->writeTabs();
			m_stream << " * Destruction\n";
			this->writeTabs();
			m_stream << " */\n";
			this->writeTabs();
			m_stream << "virtual ~" << m_name << "() { }\n\n";
		break;
		case e_template:
			this->writeTabs();
			m_stream << "/**\n";
			this->writeTabs();
			m_stream << " * Destruction\n";
			this->writeTabs();
			m_stream << " */\n";
			this->writeTabs();
			m_stream << "virtual ~" << m_name << "();\n";
			this->writeTabs();
			m_stream << "{\n";

			if (!(m_parentType == e_nothing || m_parentType == e_iCountedObject))
			{
				m_currentTabIndent++;
				this->writeTabs();
				m_currentTabIndent--;
				m_stream << "EXPONENT_CLASS_DESTRUCTION(" << m_name << "<TypeName>);\n";
			}
			m_currentTabIndent++;
			this->writeTabs();
			m_currentTabIndent--;
			m_stream << "// TODO: FILL THIS IN!\n";
			this->writeTabs();
			m_stream << "}\n\n";
		break;
		case e_struct:
			// We dont need a destructor
			m_stream << "\n";
		break;
	}
	if (!(m_parentType == e_nothing || m_parentType == e_iCountedObject))
	{
		m_stream << "//	===========================================================================\n\n";
	}
	m_currentTabIndent--;
}

//	===========================================================================
void CClassWriter::writeCountedObjectFunctions()
{
	m_currentTabIndent++;
	switch(m_parentType)
	{
		case e_nothing:
			// No functions attatched to this
		break;
		case e_iCountedObject:
			// Nothing to implement?
		break;
		case e_cCountedObject:
		case e_control:
		case e_controlPanel:
			this->writeTabs();
			m_stream << "/**\n";
			this->writeTabs();
			m_stream << " * Get a description of the object\n";
			this->writeTabs();
			m_stream << " * @param string On return is filled with the description\n";
			this->writeTabs();
			m_stream << " * @param size The size of the string\n";
			this->writeTabs();
			m_stream << " */\n";
			this->writeTabs();

			if (m_classType == e_template || m_classType == e_interface)
			{
				m_stream << "virtual void getObjectDescription(char *string, const long size) const\n";
				this->writeTabs();
				m_stream << "{\n";
				m_currentTabIndent++;
				this->writeTabs();
				m_currentTabIndent--;
				m_stream << " // TODO : FILL THIS IN!\n";
				this->writeTabs();
				m_stream << "}\n\n";
			}
			else
			{
				m_stream << "virtual void getObjectDescription(char *string, const long size) const;\n\n";
			}
			if (m_parentType == e_control)
			{
				m_stream << "//	===========================================================================\n\n";
			}
		break;
	}
	m_currentTabIndent--;
}

//	===========================================================================
void CClassWriter::writeControlFunctions()
{
	if (m_parentType != e_control)
	{
		return;
	}
	m_currentTabIndent++;

	this->writeTabs();
	m_stream << "/**\n";
	this->writeTabs();
	m_stream << " * Draw the controls\n";
	this->writeTabs();
	m_stream << " * @param graphics The graphics context\n";
	this->writeTabs();
	m_stream << " */\n";
	this->writeTabs();
	if (m_classType == e_template || m_classType == e_interface)
	{
		m_stream << "virtual void drawControl(CGraphics &graphics);\n";
		this->writeTabs();
		m_stream << "{\n";
		m_currentTabIndent++;
		this->writeTabs();

		m_stream << "// First check if we can allow the standard handler to draw the disabled control\n";
		this->writeTabs();
		m_stream << "if (!this->drawEnabledControl(graphics))\n";
		this->writeTabs();
		m_stream << "{\n";
		this->writeTabs();
		m_stream << "\treturn;\n";
		this->writeTabs();
		m_stream << "}\n\n";
		this->writeTabs();
		m_stream << "// Draw the background\n";
		this->writeTabs();
		m_stream << "this->drawPrimaryImage(graphics, m_doDefaultDrawing);\n\n";
		this->writeTabs();
		m_stream << " // TODO : FILL THIS IN!\n";
		m_currentTabIndent--;
		this->writeTabs();
		m_stream << "}\n\n";
	}
	else
	{
		m_stream << "virtual void drawControl(CGraphics &graphics);\n\n";
	}
	m_stream << "//	===========================================================================\n\n";
	this->writeTabs();
	m_stream << "/**\n";
	this->writeTabs();
	m_stream << " * Handle left button being clicked\n";
	this->writeTabs();
	m_stream << " * @param event The event to handle\n";
	this->writeTabs();
	m_stream << " */\n";
	this->writeTabs();
	if (m_classType == e_template || m_classType == e_interface)
	{
		m_stream << "virtual void handleLeftButtonDown(CMouseEvent &event);\n";
		this->writeTabs();
		m_stream << "{\n";
		m_currentTabIndent++;
		this->writeTabs();
		m_currentTabIndent--;
		m_stream << " // TODO : FILL THIS IN!\n";
		this->writeTabs();
		m_stream << "}\n\n";
	}
	else
	{
		m_stream << "virtual void handleLeftButtonDown(CMouseEvent &event);\n\n";
	}
	this->writeTabs();
	m_stream << "/**\n";
	this->writeTabs();
	m_stream << " * Handle left button being released\n";
	this->writeTabs();
	m_stream << " * @param event The event to handle\n";
	this->writeTabs();
	m_stream << " */\n";
	this->writeTabs();
	if (m_classType == e_template || m_classType == e_interface)
	{
		m_stream << "virtual void handleLeftButtonUp(CMouseEvent &event);\n";
		this->writeTabs();
		m_stream << "{\n";
		m_currentTabIndent++;
		this->writeTabs();
		m_currentTabIndent--;
		m_stream << " // TODO : FILL THIS IN!\n";
		this->writeTabs();
		m_stream << "}\n\n";
	}
	else
	{
		m_stream << "virtual void handleLeftButtonUp(CMouseEvent &event);\n\n";
	}
	this->writeTabs();
	m_stream << "/**\n";
	this->writeTabs();
	m_stream << " * Handle a double click on the left button\n";
	this->writeTabs();
	m_stream << " * @param event The event to handle\n";
	this->writeTabs();
	m_stream << " */\n";
	this->writeTabs();
	if (m_classType == e_template || m_classType == e_interface)
	{
		m_stream << "virtual void handleDoubleClick(CMouseEvent &event);\n";
		this->writeTabs();
		m_stream << "{\n";
		m_currentTabIndent++;
		this->writeTabs();
		m_currentTabIndent--;
		m_stream << " // TODO : FILL THIS IN!\n";
		this->writeTabs();
		m_stream << "}\n\n";
	}
	else
	{
		m_stream << "virtual void handleDoubleClick(CMouseEvent &event);\n\n";
	}
	this->writeTabs();
	m_stream << "/**\n";
	this->writeTabs();
	m_stream << " * Handle the mouse movement\n";
	this->writeTabs();
	m_stream << " * @param event The event to handle\n";
	this->writeTabs();
	m_stream << " */\n";
	this->writeTabs();

	if (m_classType == e_template || m_classType == e_interface)
	{
		m_stream << "virtual void handleMouseMovement(CMouseEvent &event);\n";
		this->writeTabs();
		m_stream << "{\n";
		m_currentTabIndent++;
		this->writeTabs();
		m_currentTabIndent--;
		m_stream << " // TODO : FILL THIS IN!\n";
		this->writeTabs();
		m_stream << "}\n\n";
	}
	else
	{
		m_stream << "virtual void handleMouseMovement(CMouseEvent &event);\n\n";
	}

	m_currentTabIndent--;
}

//	===========================================================================
void CClassWriter::writeProtectedOpener()
{
	if (m_classType == e_struct)
	{
		this->writeTabs();
		m_stream << "};\n";
		return;
	}
	m_stream << "//	===========================================================================\n\n";
	this->writeTabs();
	m_stream << "protected:\n\n";
	m_stream << "//	===========================================================================\n\n";
	this->writeTabs();
	m_stream << "};\n";
}

//	===========================================================================
void CClassWriter::writeTemplateExtras()
{
	if (m_classType == e_template && (m_parentType == e_cCountedObject || m_parentType == e_control || m_parentType == e_controlPanel))
	{
		CString parentName;
		switch(m_parentType)
		{
			case e_cCountedObject:	parentName = "CCountedObject";			break;
			case e_control:			parentName = "CControl";				break;
			case e_controlPanel:	parentName = "CControlPanel";			break;
		}
		m_stream << "\n";
		this->writeTabs();
		m_stream << "/** @cond */\n";
		this->writeTabs();
		m_stream << "EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(" << m_name << "<TypeName>, TypeName, " << parentName << ");\n";
		this->writeTabs();
		m_stream << "/** @endcond */\n";
	}
}

//	===========================================================================
void CClassWriter::writeImplementationFile()
{
	// Dont have impl files
	if (m_classType == e_template || m_classType == e_struct || m_classType == e_interface)
	{
		// These dont have cpp files
		return;
	}

	// Construct the header file path
	CSystemString implementationPath = m_outputPath;
	implementationPath.appendPath(m_name);
	implementationPath.appendNewExtension("cpp");

	// Open the stream
	if (!m_stream.openStream(implementationPath, CTextStream::e_output))
	{
		CDialog::notifyUser("Failed to open stream to write implementation file", "ClassWrite error", true);
		return;
	}

	// First write the CVS comments opening
	m_stream << "// $Id: CClassWriter.cpp,v 1.6 2007/02/11 02:20:05 paul Exp $\n\n";
	CTime myTime;
	m_stream << "/*\n * All contents of this source code are copyright " << myTime.getYear() << " Exp Digital Uk.\n * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n * with the source code. If you didnt, please refer to http://www.expdigital.co.uk\n * All content is the Intellectual property of Exp Digital Uk.\n * Certain sections of this code may come from other sources. They are credited where applicable.\n * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk\n  */\n\n";
	m_stream << "#include \"" << m_name << ".hpp\"\n";

	// Writ ethe namespace
	m_stream << "using ";
	for (long i = 0; i < m_namespace.getArraySize(); i++)
	{
		const CString *space = m_namespace.constElementAtIndex(i);

		if (space == NULL)
		{
			continue;
		}

		m_stream << *space;

		if (i != m_namespace.getArraySize() - 1)
		{
			m_stream << "::";
		}
		else
		{
			m_stream << "::" << m_name << ";\n\n";
		}
	}

	if (m_parentType == e_cCountedObject || m_parentType == e_control || m_parentType == e_controlPanel)
	{
		CString parentName;
		switch(m_parentType)
		{
			case e_cCountedObject:		parentName = "CCountedObject";			break;
			case e_control:				parentName = "CControl";				break;
			case e_controlPanel:		parentName = "CControlPanel";			break;
		}
		m_stream << "//	===========================================================================\n";
		m_stream << "EXPONENT_CLASS_IMPLEMENTATION(" << m_name << ", " << parentName << ");\n\n";
	}

	// Write the constructor
	m_stream << "//	===========================================================================\n";
	m_stream << m_name << "::" << m_name << "(";

	if (m_parentType == e_control)
	{
		m_stream << "IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener)\n";
		long space = m_name.getNumberOfCharacters() - 1;
		for (long i = 0; i < space; i++)
		{
			m_stream << " ";
		}
		m_stream << ": CControl(root, uniqueId, area, listener)\n";
		m_stream << "{\n\tEXPONENT_CLASS_CONSTRUCTION(" << m_name << ");\n\t// TODO: FILL THIS IN!\n}\n\n";
	}
	else if (m_parentType == e_controlPanel)
	{
		m_stream << "IWindow *parent, IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener)\n";
		long space = m_name.getNumberOfCharacters() - 1;
		for (long i = 0; i < space; i++)
		{
			m_stream << " ";
		}
		m_stream << ": CControlPanel(root->getParentWindow(), root, uniqueId, area, listener)\n";
		m_stream << "{\n\tEXPONENT_CLASS_CONSTRUCTION(" << m_name << ");\n\t// TODO: FILL THIS IN!\n}\n\n";
	}
	else
	{
		if (m_parentType == e_nothing || m_parentType == e_iCountedObject)
		{
			m_stream << ")\n{\n\t// TODO: FILL THIS IN!\n}\n\n";
		}
		else
		{
			m_stream << ")\n{\n\tEXPONENT_CLASS_CONSTRUCTION(" << m_name << ");\n\t// TODO: FILL THIS IN!\n}\n\n";
		}
	}

	m_stream << "//	===========================================================================\n";

	if (m_parentType == e_nothing || m_parentType == e_iCountedObject)
	{
		m_stream << m_name << "::~" << m_name << "()\n{\n\t// TODO : FILL THIS IN!\n}\n\n";
	}
	else
	{
		m_stream << m_name << "::~" << m_name << "()\n{\n\tEXPONENT_CLASS_DESTRUCTION(" << m_name << ");\n\t// TODO : FILL THIS IN!\n}\n\n";
		m_stream << "//	===========================================================================\n";
		m_stream << "void " << m_name << "::getObjectDescription(char *string, const long size) const\n{\n\t// TODO : FILL THIS IN!\n}\n\n";
	}

	if (m_parentType == e_control)
	{
		m_stream << "//	===========================================================================\n";
		m_stream << "void " << m_name << "::drawControl(CGraphics &graphics)\n{\n\t// First check if we can allow the standard handler to draw the disabled control\n\tif (!this->drawEnabledControl(graphics))\n\t{\n\t\treturn;\n\t}\n\n\t// Draw the background\n\tthis->drawPrimaryImage(graphics, m_doDefaultDrawing);\n\n\t// TODO : FILL THIS IN!\n}\n\n";
		m_stream << "//	===========================================================================\n";
		m_stream << "void " << m_name << "::handleLeftButtonDown(CMouseEvent &event)\n{\n\t// TODO : FILL THIS IN!\n}\n\n";
		m_stream << "//	===========================================================================\n";
		m_stream << "void " << m_name << "::handleLeftButtonUp(CMouseEvent &event)\n{\n\t// TODO : FILL THIS IN!\n}\n\n";
		m_stream << "//	===========================================================================\n";
		m_stream << "void " << m_name << "::handleDoubleClick(CMouseEvent &event)\n{\n\t// TODO : FILL THIS IN!\n}\n\n";
		m_stream << "//	===========================================================================\n";
		m_stream << "void " << m_name << "::handleMouseMovement(CMouseEvent &event)\n{\n\t// TODO : FILL THIS IN!\n}";
	}
}